﻿/*
  URL Cert Key Provider for KeePass 
  Copyright (C) 2012 Dirk Heitzmann <URLCertKeyProvider (a-t) c-wd.de>

  * URLCertKeyProviderClass 
  * Class for capsulate the functionality and methodes 

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  In addition :
    Uncommercial, personnel use is free.
    For commercial use see Copyright.
    Removing of information about Copyright is prohibited.
*/
using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Windows.Forms;
using System.Xml.Linq;

namespace URLCertKeyProvider
{
    public class URLCertKeyProviderClass
    {
        //----------------------
        public struct urlfileStruct
        {
            public XDocument Xdoc { get; set; }
            public XElement Xroot { get; set; }
            public String filename { get; set; }
            public String url { get; set; }
        }
        public urlfileStruct urlfile;
        
        public struct keydocumentStruct
        {
            public String XMLString { get; set; }
            public XDocument Xdoc { get; set; }
            public XElement Xroot { get; set; }
        }
        public keydocumentStruct keydocument;

        //----------------------
        public struct AESKeyStruct
        {
            public string Base64 { get; set; }
            public string Hash { get; set; }
            public byte[] Key { get; set; }
        }
        public AESKeyStruct AESKey;
        public AESKeyStruct AESKeyDecrypted;

        //----------------------
        private X509Certificate2 p_DecryptRSACert;
        public X509Certificate2 DecryptRSACert
        {
            get { return p_DecryptRSACert; }
            set { p_DecryptRSACert = value; }
        }


        //----------------------
        public Boolean statURLFileOpened   { get{return (urlfile.url.Length > 0);} }
        public Boolean statURLisWellFormed { get{return Uri.IsWellFormedUriString(urlfile.url, UriKind.Absolute);} }
        public Boolean statKeysFetched     { get{return (keydocument.XMLString.Length > 0);} }
        public Boolean statKeysParsed      { get{return (keydocument.Xroot.HasElements);} }
        public Boolean statAESDecrypted    { get{return (AESKeyDecrypted.Key != null);} }

        
        public Boolean OpenURLfile() 
        {
            InitAESStructures();
            urlfile.Xdoc = XDocument.Load(urlfile.filename);
            urlfile.Xroot = urlfile.Xdoc.Element("CryptedKeys");
            urlfile.url = urlfile.Xroot.Element("URL").Value;
            return true;
        }
        public Boolean OpenURLfile(string _filename)
        {
            urlfile.filename = _filename;
            return OpenURLfile();
        }
 
        public Boolean CreateURLfile()
        {
            if (urlfile.filename.Length == 0)
                throw new FileNotFoundException("ERROR : URLfile - Filename not set.");

            urlfile.Xdoc = new XDocument(
                new XDeclaration("1.0", "utf-8", "yes"),
                new XComment("URLfile for UrlCertKeyProvider, (c) Dirk Heitzmann, creativeit.eu"),
                new XElement("CryptedKeys",
                    new XElement("URL", urlfile.url)
                )
            );
            urlfile.Xdoc.Save(urlfile.filename);
            return true;
        }
        public Boolean CreateURLfile(string _filename)
        {
            urlfile.filename = _filename;
            return CreateURLfile();
        }
 
        //----------------------
        public Boolean GetKeysFromWeb()
        {
            X509Certificate2Collection certCollection = null;

            // Prerequisites
            if (!Uri.IsWellFormedUriString(urlfile.url, UriKind.Absolute))
                throw new ArgumentOutOfRangeException("ERROR : URL is not an absolute URL (Uri.IsWellFormedUriString,UriKind.Absolute).");

            keydocument.XMLString = "";
            InitAESStructures();

            // Prepare the X509Certificate Collection
            certCollection = new X509Certificate2Collection(p_DecryptRSACert);

            // Request the key data.
            HttpWebRequest Request = (HttpWebRequest)WebRequest.Create(urlfile.url);
            Request.ClientCertificates = certCollection;
            Request.UserAgent = "KeePass URLCertKeyProvider";
            Request.Method = "POST";

            // Process the response, the key data.
            HttpWebResponse Response = (HttpWebResponse)Request.GetResponse();
            StreamReader sr = new StreamReader(Response.GetResponseStream(), Encoding.Default);

            keydocument.XMLString = sr.ReadToEnd();
            return true;
        }
        public Boolean GetKeysFromWeb(string _Url)
        {
            urlfile.url = _Url;
            return GetKeysFromWeb();
        }

        public Boolean ParseKeys(Boolean doDecrypt = true) 
        {
            if (keydocument.XMLString.Length == 0)
                throw new ArgumentOutOfRangeException("ERROR : XMLString is empty.");

            InitAESStructures();

            keydocument.Xdoc = XDocument.Parse(keydocument.XMLString);
            keydocument.Xroot = keydocument.Xdoc.Element("CryptedKeys");

            if (keydocument.Xroot.Element("Error").Value != null)
                throw new Exception("Response" + keydocument.Xroot.Element("Error").Value);

            AESKey.Hash = keydocument.Xroot.Attribute("KeyHash").Value;

            if (doDecrypt)
            {
                if (DecryptAESKeyWithRSA())
                    return true;
                else
                    return false;
            }
            else
                return keydocument.Xroot.HasElements;
        }
        public Boolean ParseKeys(string response, Boolean doDecrypt = true)
        {
            keydocument.XMLString = response;
            return ParseKeys(doDecrypt);
        }

        private void InitAESStructures()
        {
            AESKeyDecrypted.Base64 = "";
            AESKeyDecrypted.Hash = "";
            AESKeyDecrypted.Key = null;
            AESKey.Base64 = "";
            AESKey.Hash = "";
            AESKey.Key = null;
        }

        public Boolean DecryptAESKeyWithRSA()
        {
            return DecryptAESKeyWithRSA(p_DecryptRSACert);
        }
        public Boolean DecryptAESKeyWithRSA(X509Certificate2 decryptCert)
        {
            XElement p_XDecryptedKey = null;
            String p_XDecryptedKey_Key = "";

            InitAESStructures();
            try
            {
                XElement p_key = keydocument.Xroot.Descendants("Keyname")
                         .Where(n => (string)n == decryptCert.Subject)
                         .FirstOrDefault();
                p_XDecryptedKey = p_key.Ancestors("CryptedKey").FirstOrDefault();
                p_XDecryptedKey_Key = p_XDecryptedKey.Element("Key").Value;
                
                if (p_XDecryptedKey != null && p_XDecryptedKey.HasElements)
                {
                    AESKeyDecrypted.Base64 = RSADecryptKey(p_XDecryptedKey_Key, decryptCert);
                    AESKeyDecrypted.Hash = GetMD5Hash(AESKeyDecrypted.Base64);

                    AESKey.Base64 = AESKeyDecrypted.Base64;

                    if (AESKeyDecrypted.Hash.Equals(AESKey.Hash))
                    {
                        AESKeyDecrypted.Key = Convert.FromBase64String(ASCIIEncoding.UTF8.GetString(Convert.FromBase64String(AESKeyDecrypted.Base64)).Split(',')[1]);
                        AESKey.Key = AESKeyDecrypted.Key;
                        return true;
                    }
                    return false;
                }
                else
                {
                    return false;
                }
            }
            catch (InvalidOperationException)
            {
                return false;
            }
        }

        // ------------------------------------------------------
        // RSA
        public Boolean RSASelectCertificate()
        {
            X509Store store = new X509Store("MY", StoreLocation.CurrentUser);
            store.Open(OpenFlags.ReadOnly | OpenFlags.OpenExistingOnly);

            X509Certificate2Collection collection = (X509Certificate2Collection)store.Certificates;
            X509Certificate2Collection fcollection = (X509Certificate2Collection)collection.Find(X509FindType.FindByTimeValid, DateTime.Now, false);
            fcollection = (X509Certificate2Collection)fcollection.Find(X509FindType.FindByKeyUsage, X509KeyUsageFlags.DataEncipherment, true);
            X509Certificate2Collection scollection = X509Certificate2UI.SelectFromCollection(fcollection, "Certificate Select", "Select a certificate from the following list", X509SelectionFlag.SingleSelection);

            if (scollection.Count == 0)
            {
                throw new Exception("ERROR : No certificate selected.");
            }
            if (!scollection[0].HasPrivateKey)
            {
                throw new CryptographicException("ERROR : Certificate contains no private key.");
            }

            p_DecryptRSACert = scollection[0];
            store.Close();

            return true;
        }
        public String RSADecryptKey(String String2Decrypt, X509Certificate2 certificate)
        {
            RSACryptoServiceProvider rsa = certificate.PrivateKey as RSACryptoServiceProvider;

            byte[] cipherbytes = Convert.FromBase64String(String2Decrypt);
            byte[] plainbytes = rsa.Decrypt(cipherbytes, false);

            System.Text.ASCIIEncoding enc = new System.Text.ASCIIEncoding();
            return enc.GetString(plainbytes);
        }

        // ------------------------------------------------------
        // MD5 Stuff
        public string GetMD5Hash()
        {
            return GetMD5Hash(AESKeyDecrypted.Base64);
        }
        public string GetMD5Hash(string TextToHash)
        {
            //Prüfen ob Daten übergeben wurden.
            if ((TextToHash == null) || (TextToHash.Length == 0))
            {
                return string.Empty;
            }
            //MD5 Hash aus dem String berechnen. Dazu muss der string in ein Byte[]
            //zerlegt werden. Danach muss das Resultat wieder zurück in ein string.
            MD5 md5 = new MD5CryptoServiceProvider();
            byte[] textToHash = Encoding.Default.GetBytes(TextToHash);
            byte[] result = md5.ComputeHash(textToHash);

            AESKey.Hash = System.BitConverter.ToString(result);
            return AESKey.Hash;
        }

        public void DoCreate()
        {
            URLCertKeyProviderDialog frmTest = new URLCertKeyProviderDialog();
            frmTest.KeyProvider = this;

            URLCertKeyProviderProgessFrom frmProgress = new URLCertKeyProviderProgessFrom();

            this.urlfile.url = "https://yourServerORwebspace/KeymanagerFolderThere/GetKeys.php";

            if (frmTest.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    frmProgress.Show();

                    frmProgress.actionText = "Opening the URLFile";
                    frmProgress.actionValue = 25;
                    System.Threading.Thread.Sleep(500);

                    if (this.CreateURLfile())
                    {
                        frmProgress.actionText = "Getting the encrypted key from the URL";
                        frmProgress.actionValue = 50;
                        System.Threading.Thread.Sleep(500);
                        
                        if (this.GetKeysFromWeb())
                        {
                            frmProgress.actionText = "Parsing the encrypted key";
                            frmProgress.actionValue = 75;
                            System.Threading.Thread.Sleep(500);
                            
                            if (!this.ParseKeys())
                                throw new CryptographicException("Unable to parse and decrypt the key.");
                        }
                        else
                            throw new FileNotFoundException("Unable to fetch key.");
                    }
                    else
                        throw new FileNotFoundException("Unable to create the new keyfile.");
                }
                catch (Exception ex)
                {
                    throw (ex);
                }
                finally
                {
                    frmProgress.actionText = "D O N E";
                    frmProgress.actionValue = 100;
                    System.Threading.Thread.Sleep(500);
                    
                    frmProgress.Close();
                    frmProgress = null;
                    frmTest = null;
                }
            }
        }

        public void DoExisting()
        {
            URLCertKeyProviderProgessFrom frmProgress = new URLCertKeyProviderProgessFrom();

            // get the certificate 
            this.RSASelectCertificate();

            try
            {
                frmProgress.Show();

                frmProgress.actionText = "Opening the URLFile";
                frmProgress.actionValue = 25;
                System.Threading.Thread.Sleep(500);

                if (this.OpenURLfile()) {
                    frmProgress.actionText = "Getting the encrypted key from the URL";
                    frmProgress.actionValue = 50;
                    System.Threading.Thread.Sleep(500);

                    if (this.GetKeysFromWeb())
                    {
                        frmProgress.actionText = "Parsing the encrypted key";
                        frmProgress.actionValue = 75;
                        System.Threading.Thread.Sleep(500);

                        if (!this.ParseKeys())
                            throw new CryptographicException("Unable to parse and decrypt the key.");
                    }
                    else
                        throw new FileNotFoundException("Unable to fetch key.");
                }
                else
                    throw new FileNotFoundException("Unable to load keyfile.");
            }
            catch (Exception ex)
            {
                throw(ex);
            }
            finally
            {
                frmProgress.actionText = "D O N E";
                frmProgress.actionValue = 100;
                System.Threading.Thread.Sleep(500);

                frmProgress.Close();
                frmProgress=null;
            }
        }

    }
}
